#!/usr/bin/env python3
"""Entry point for the string‑tension multi‑gauge low sweep integration.

This driver is designed to be invoked by the Fractal‑Pivot Hypersurface (FPH)
orchestrator.  It parses the parent YAML configuration, resolves all file paths
relative to the repository root and delegates the actual parameter sweep to
``sweep_low_gauge.main``.  The helper scripts ``compute_Amu.py``,
``compute_Umu.py`` and ``measure_wilson.py`` are bundled in this package
but are not used directly here – the sweep driver uses real data from local data/ or parent config.
"""

from __future__ import annotations

import argparse
import os
import sys
import yaml

# Import the sweep driver.  Use an absolute import so this module can
# also be executed as a script without requiring the package context.
# Ensure the repository root is on the Python path so that absolute imports
# of this package succeed when the script is executed directly via ``python run.py``.
# The repository root sits one directory above this package.  ``__file__``
# resolves to ``.../repo/vol4_string_tension_multi_gauge_low_sweep/run.py`` so
# joining with one ``os.pardir`` yields ``.../repo``.
repo_root_for_sys = os.path.abspath(os.path.join(os.path.dirname(__file__), os.pardir))
if repo_root_for_sys not in sys.path:
    sys.path.insert(0, repo_root_for_sys)

from vol4_string_tension_multi_gauge_low_sweep.sweep_low_gauge import main as sweep


def main() -> None:
    parser = argparse.ArgumentParser(description="Run string‑tension multi‑gauge low sweep")
    parser.add_argument("--config", required=True, help="Path to the YAML configuration file")
    parser.add_argument("--output-dir", required=True, help="Directory for outputs")
    args = parser.parse_args()

    # Load configuration from parent
    with open(args.config) as f:
        cfg = yaml.safe_load(f)

    # Resolve repository root (two levels above this file)
    repo_root = os.path.abspath(os.path.join(os.path.dirname(__file__), os.pardir, os.pardir))

    # Use local standalone data/ for kernel and flip‑counts (absolute path from your address)
    local_data_dir = r'C:\Users\kentn\vol4-fractal-pivot-hypersurface\vol4_string_tension_multi_gauge_low_sweep\data'
    kernel_path = os.path.join(local_data_dir, 'kernel.npy')
    flip_counts_path = os.path.join(local_data_dir, 'flip_counts.npy')

    if not os.path.exists(kernel_path) or not os.path.exists(flip_counts_path):
        raise FileNotFoundError("Local real kernel.npy or flip_counts.npy missing in data/")

    # Override kernel and flip‑count paths to use the module's bundled data.
    # This ensures that the sweep uses real data shipped with this package.  Each
    # gauge group is mapped to the same file because the sweep driver does
    # not distinguish between U1, SU2 and SU3 kernels.  For a real simulation
    # the kernels would be separate files with appropriate dimensions.
    cfg['kernel_paths'] = {
        'U1': kernel_path,
        'SU2': kernel_path,
        'SU3': kernel_path,
    }
    cfg['flip_counts_path'] = flip_counts_path

    # Delegate to the sweep driver
    sweep(cfg, args.output_dir)


if __name__ == '__main__':
    main()